/*
 * mqtt_network_Manager.cpp
 *
 *  Created on: Oct 21, 2018
 *      Author: gjd5kor
 */





#include "mqtt_network_Manager.h"
#include "mqtt_Class_Types.h"
#include "stunnel_Handler.h"
#include "stunnel_log_parser.h"
#include "mqtt_DLT_Trace_Header.h"
#include "DataCompressionHandler.h"
#include "Imqtt_manager.h"
#include "I_mqtt_log_Manager.h"
#include "I_mqtt_JSONParser.h"
#include <sstream>


#define MOSQ_ERR_CONNACK_NOT_RECIEVED 254
#define PUBLISH_QOS 1
#define SUBSCRIBE_QOS 1
#define PUBLISH_RETAIN true
#define PENDING_PUBLISH_ITEM_POST_DELAY 2000

#define TRACE_FILE_NAME "mqtt_network_Manager.cpp"

void mqtt_network_Manager::onExpired(Timer& timer,::boost::shared_ptr<TimerPayload> data)
{
	if(&timer == &oPubACKTimer)
	{
		MQTT_LOG_INFO_VAL(("PUBACK TIMER EXPIRED"));
		ActionOnPUBACKTimerExpired();
	}

	if(&timer == &oSubAckTimer)
	{
		MQTT_LOG_INFO_VAL(("SUBACk TIMER EXPIRED"));
		ActionOnSUBACKTimerExpired();
	}

	if(&timer == &oConACKTimer)
	{
		MQTT_LOG_INFO_VAL(("CONACK TIMER EXPIRED"));
		ActionOnCONACKTimerExpired();
	}

	if(&timer == &oPublishIntervalTimer)
	{
		MQTT_LOG_INFO_VAL(("PUBLISH INTERVAL EXPIRED"));
		ActionOnPUBINTimerExpired();
	}

	if(&timer == &oPendingItemUploadTimer)
	{
		MQTT_LOG_INFO_VAL(("TRIGGERING PENDING PUBLISH ITEM TIMER EXPIRED"));
		RetryPublish();
	}


}

mqtt_network_Manager::mqtt_network_Manager(mqtt_manager_tclAppMain *_poMainAppl,std::string &sNaviId,bool CS,bool _CompressionState)
:mqtt_MosquittoCppWrapper(sNaviId.c_str(),CS),I_mqtt_network_Manager(_poMainAppl)
{
	MQTT_LOG_INFO_VAL(("CONSTRUCTOR FOR NW MANAGER CALLED"));

	int result = lib_init();

	if(MOSQ_ERR_SUCCESS != result)
	{
		MQTT_LOG_FATAL(("FAILED TO INITIALIZE MOSQUITTO LIBRARY"));
		lib_cleanup();
	}

	bIpServiceAvaibility = false;
	NetworkLoopActive = false;
	Connected = false;;
	Compression_State = _CompressionState;
	Initialized = false;
	bReconnectedDueToPublishFail = false;
	Remaining_Retry_Attempts = 0;
	DisconnectionReason = UNKNOWN;
	keepaliveval = 0;
	Publish_Occurence = 0;
	Publish_Timeout = 0;
	Publish_Interval = 0;
}

bool mqtt_network_Manager::Initialize(std::string& sNaviId,bool CS, bool _CompressionMode)
{
	MQTT_LOG_INFO_VAL(("INITIALIZING THE MQTT INSTANCE"));
	MQTT_LOG_INFO_VAL(("CLIENT_ID :: %s",sNaviId.c_str()));
	MQTT_LOG_INFO_VAL(("CLEAN_SESSION :: %d",CS));
	MQTT_LOG_INFO_VAL(("COMPRESSION_MODE :: %d",_CompressionMode));

	Compression_State = _CompressionMode;

	LoadTLSErrorList();

	sNaviUnitId = sNaviId;
	reinitialise(sNaviId.c_str(),CS);


	MQTT_LOG_INFO_VAL(("SETTING MAX RECONNECT DELAY "));
	reconnect_delay_set(UINT32_MAX,UINT32_MAX,false);

	MQTT_LOG_INFO_VAL(("SETTING MAX MESSAGE RETRY DELAY "));
	message_retry_set(0);

	int mqtt_ver = MQTT_PROTOCOL_V311;
	void* ver = &mqtt_ver;
	opts_set(MOSQ_OPT_PROTOCOL_VERSION,ver);
	MQTT_LOG_INFO_VAL(("SETTING PROTOCOL VERSION TO MQTT_PROTOCOL_V311"));

	Initialized = true;


	MQTT_LOG_INFO_VAL(("INITIALIZED THE MOSQUITTO LIBRARY"));
	return true;
}

mqtt_network_Manager::~mqtt_network_Manager()
{
	lib_cleanup();
}


void mqtt_network_Manager::ActionOnCONACKTimerExpired()
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::ActionOnCONACKTimerExpired() entered"));
	MQTT_LOG_INFO_VAL(("CONACK NOT RECIEVED"));
	ActionOnDisconnectionResponse(MOSQ_ERR_CONNACK_NOT_RECIEVED);
}


void mqtt_network_Manager::on_connect(int rc)
{
//	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_connect() entered"));
//	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %d",rc));
//	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %s",mosquitto_connack_string(rc)));
}

void mqtt_network_Manager::on_connect_with_flags(int rc, int flags)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_connect_with_flags() entered"));
	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %d",rc));
	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %s",mosquitto_connack_string(rc)));
	MQTT_LOG_INFO_VAL(("SESSION RESUMPTION FLAG :: %d",flags));

	MQTT_LOG_INFO_VAL(("STOPPING CONACK TIMER"));
	oConACKTimer.stop();

	if(MOSQ_ERR_SUCCESS == rc)
	{
		MQTT_LOG_INFO_VAL((" CONNECTED TO BROKER  :)"));
		Connected = true;

		if(true ==  bReconnectedDueToPublishFail)
		{
			PublishRetryOnConnection();
		}

		MqttConnectionResponse *Response = new MqttConnectionResponse(CONNECTION);
		Response->setConnectionResponseCode(CONNECTED);
		Response->setSessionResumption((unsigned short)flags);
		SendResponse(Response);
	}

}

void mqtt_network_Manager::on_disconnect(int rc)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_disconnect() entered"));
	MQTT_LOG_INFO_VAL(("DISCONNECT RESPONSE :: %d",rc));
	MQTT_LOG_INFO_VAL(("DISCONNECT RESPONSE :: %s",mosquitto_connack_string(rc)));
	oConACKTimer.stop();
	ActionOnDisconnectionResponse(rc);
}

void mqtt_network_Manager::on_publish(int mid)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_publish() entered"));
	MQTT_LOG_INFO_VAL(("PUBLISHED MESSAGE ID :: %d",mid));
	bReconnectedDueToPublishFail = false;
	Remaining_Retry_Attempts = 0;
	oPubACKTimer.stop();
	MQTT_LOG_INFO_VAL(("bReconnectedDueToPublishFail:: %d",bReconnectedDueToPublishFail));
	I_mqtt_log_Manager*_poLogMgr = dynamic_cast<I_mqtt_log_Manager*>(_cpoMain->getHandler("I_mqtt_log_Manager"));
	MQTT_MANAGER_NULL_POINTER_CHECK(_poLogMgr);
	_poLogMgr->vWriteLogToFile(UPLOAD,"STAGE 3 : PUBLISHED","UploadData.log",false);
	MqttPublishResonse *Response = new MqttPublishResonse(PUBLISH);
	Response->setMid(mid);
	Response->setResponseCode(PUBLISHED);
	SendResponse(Response);
}

void mqtt_network_Manager::on_message(const struct mosquitto_message* message)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_message() entered"));
	MQTT_LOG_INFO_VAL(("***** NEW MESSAGE RECIEVED ******"));
	MQTT_LOG_INFO_VAL((" MID :: %d ",message->mid));
	MQTT_LOG_INFO_VAL((" TOPIC :: %s ",message->topic));
	MQTT_LOG_INFO_VAL((" PAYLOAD LENGTH :: %d ",message->payloadlen));
	MQTT_LOG_INFO_VAL((" QOS :: %d ",message->qos));
	MQTT_LOG_INFO_VAL((" RETAIN :: %d ",message->retain));
	MQTT_LOG_INFO_VAL((" PAYLOAD  :: %s ",(char*)message->payload));
	MQTT_LOG_INFO_VAL(("***** END OF MESSAGE RECIEVED ******"));
	const char* _str = static_cast<const char*>(message->payload);
	std::string sTmp( reinterpret_cast<char const*>(_str), message->payloadlen);
	std::string sTopic(message->topic);
	MqttMessage *Message = new MqttMessage(MESSAGE);
	Message->setMid(message->mid);
	Message->setPayload(sTmp);
	Message->setTopic(sTopic);
	Message->setRetain(message->retain);
	SendResponse(Message);
}

void mqtt_network_Manager::on_subscribe(int mid, int qos_count,	const int* granted_qos)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_subscribe() entered"));
	MQTT_LOG_INFO_VAL((" SUBSCIBED MSG ID :: %d",mid));
	MQTT_LOG_INFO_VAL((" QOS COUNT :: %d",qos_count));
	//MQTT_LOG_INFO_VAL((" GRANTED QOS :: %d", granted_qos));
	oSubAckTimer.stop();
	MqttSubscribeResponse *Response = new MqttSubscribeResponse(SUBSCRIBE);
	Response->setResponseCode(SUBSCRIBED);
	Response->setMid(mid);
	Response->setGrantedQos(*granted_qos);
	SendResponse(Response);
}

void mqtt_network_Manager::on_unsubscribe(int mid)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_unsubscribe() entered"));
	MQTT_LOG_INFO_VAL((" UNSUBSCIBED MSG ID :: %d",mid));
}

void mqtt_network_Manager::on_log(int level, const char* str)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_log() entered"));
	MQTT_LOG_INFO_VAL((" LOG LEVEL:: %d",level));
	MQTT_LOG_INFO_VAL((" LOG MSG:: %s",str));
	I_mqtt_log_Manager*_poLogMgr = dynamic_cast<I_mqtt_log_Manager*>(_cpoMain->getHandler("I_mqtt_log_Manager"));
	MQTT_MANAGER_NULL_POINTER_CHECK(_poLogMgr);
	_poLogMgr->vWriteLogToFile(MQTT_NETWORK,str,"mqtt_network_log.txt",false);
}

void mqtt_network_Manager::on_error()
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::on_error() entered"));
}

void mqtt_network_Manager::ActionOnDisconnectionResponse(int connection_response)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::ActionOnDisconnectionResponse() entered"));
	MQTT_LOG_INFO_VAL((" DISCONNECTION CODE :: %d ",connection_response));

	std::string sDisconnectionLog = "Client ";
	sDisconnectionLog.append(sNaviUnitId);
	sDisconnectionLog.append(" received DISCONNECTED DUE TO ");

	Connected = false;
	bool loop_force = false;

	MqttConnectionResponse *response = new MqttConnectionResponse(CONNECTION);
	unsigned ConnectionResponseCode = UNKNOW_RESPONSE_CODE;
	switch(connection_response)
	{
		case MOSQ_ERR_SUCCESS:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("EXPECTED DISCONNECTION "));

			switch(DisconnectionReason)
			{
				case EXTRANAL:
				{
					ConnectionResponseCode = DISCONNECTED;
					MQTT_LOG_INFO_VAL(("DISCONNECTED DUE TO SESSIONTIMEOUT/CONNMAN NOT AVAILABLE"));
					sDisconnectionLog.append("SESSIONTIMEOUT/IP CONNECTIVITY LOST");
				}
				break;

				case SUBSCRIBE_FAIL:
				{
					ConnectionResponseCode = CONNECTION_DISCONNECTED_DUE_TO_SUBSCRIBE_FAILED;
					MQTT_LOG_INFO_VAL(("DISCONNECTED DUE TO SUBSCRIBE FAILED"));
					sDisconnectionLog.append("LAST SUBSCRIBE REQUEST FAILED");
				}
				break;

				case PUBLISH_FAIL:
				{
					ConnectionResponseCode = CONNECTION_DISCONNECTED_DUE_TO_PUBLISH_FAILED;
					MQTT_LOG_INFO_VAL(("DISCONNECTED DUE TO PUBLISH FAILED"));
					sDisconnectionLog.append("LAST PUBLISH REQUEST FAILED");
				}
				break;

				case CONACK_FAIL:
				{
					ConnectionResponseCode = CONNECTION_FAILED_NO_CONACK;
					MQTT_LOG_INFO_VAL(("DISCONNECTED DUE CONACK NOT RECIEVED"));
					sDisconnectionLog.append("CONACK NOT RECIEVED");
				}
				break;
			}
		}
		break;

		case MOSQ_ERR_CONN_REFUSED:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("NON TLS ISSUE "));
			ConnectionResponseCode = CONNECTION_FAILED_BROKER_REFUSED;
			sDisconnectionLog.append("CLIENT NOT AUTHORIZED");
		}
		break;

		case MOSQ_ERR_PROTOCOL:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("NON TLS ISSUE "));
			ConnectionResponseCode = CONNECTION_FAILED_BROKER_REFUSED;
			sDisconnectionLog.append("INVALID PROTOCOL ");
		}
		break;

		case MOSQ_ERR_INVAL:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("NON TLS ISSUE "));
			ConnectionResponseCode = CONNECTION_FAILED_BROKER_REFUSED;
			sDisconnectionLog.append("BROKER REFUSED");
		}
		break;

		case MOSQ_ERR_NOMEM:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("PING RESP NOT RECIVED "));
			ConnectionResponseCode = CONNECTION_FAILED_BROKER_REFUSED;
			sDisconnectionLog.append("PINGRESP NOT RECIEVED ");
		}
		break;

		case MOSQ_ERR_CONN_LOST:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL(("CHECK FOR TLS ISSUE "));
			ConnectionResponseCode = CONNECTION_FAILED_BROKER_DISCONNECTED;
			std::string sLog = "BROKER DISCONNECTED";

			bool isTLSFatalError = false;

			try
			{
				isTLSFatalError = bIsTLSFatalError();
			}
			catch (std::bad_alloc& ba)
			{
				MQTT_LOG_FATAL(("FAILED TO FIND TLS ERROR. BAD ALLOC :: std::bad_alloc caught :: %s",ba.what()));
			}

			if(true == isTLSFatalError)
			{
				MQTT_LOG_INFO_VAL(("TLS FATAL !!!!!!!!!!!!!!!! "));
				ConnectionResponseCode = CONNECTION_FAILED_TLS_FATAL_ERROR;
				sLog = "DUE TO TLS FATAL ERROR OCCURED";
			}
			sDisconnectionLog.append(sLog);
		}
		break;

		case MOSQ_ERR_CONNACK_NOT_RECIEVED:
		{
			loop_force = false;
			MQTT_LOG_INFO_VAL((" CONNACK NOT RECIEVED"));
			DisconnectionReason = CONACK_FAIL;
			disconnect();
			stop_loop(loop_force);
			delete response;
			return;
		}
		break;

		default:
		{
			loop_force = true;
			ConnectionResponseCode = UNKNOW_RESPONSE_CODE;
			MQTT_LOG_INFO_VAL((" RESPONSE %d IS NOT HANDLED",connection_response));
			sDisconnectionLog.append("UNKNOWN RESPONSE");
		}
		break;

	}

	AdditionalStepsOnDisconnection(ConnectionResponseCode);
	UpdateDisconnectionReasonToMqttLogFile(sDisconnectionLog,connection_response);
	response->setConnectionResponseCode(ConnectionResponseCode);
	SendResponse(response);
	if((ConnectionResponseCode == CONNECTION_FAILED_BROKER_DISCONNECTED) ||(ConnectionResponseCode == CONNECTION_FAILED_BROKER_REFUSED))
	{
		message_retry_set(UINT16_MAX);
		disconnect();
		if (Remaining_Retry_Attempts > 0)
		{
			MQTT_LOG_INFO_VAL((" BROKER DISCONNETED WHILE PUBLISH IN PROGRESS"));
			bReconnectedDueToPublishFail = true;
		}
		stop_loop(loop_force);

	}
	else
	{
		stop_loop(loop_force);
	}


}

int mqtt_network_Manager::Connect(std::string& sUrl, unsigned port,unsigned short keepAlive, unsigned short ConAckTimeOut)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Connect() entered"));
	MQTT_LOG_INFO_VAL(("CONNECT REQUEST RECIEVED"));
	int connection_response = MOSQ_ERR_EAI;

	if(false == ConnectionPreCheck(connection_response))
	{
		MQTT_LOG_INFO_VAL(("CONNECT REQUEST REFUSED"));
		return connection_response;
	}

	MQTT_LOG_INFO_VAL(("URL  :: %s",sUrl.c_str()));
	MQTT_LOG_INFO_VAL(("PORT :: %d",port));
	MQTT_LOG_INFO_VAL(("KEEPALIVE :: %d",keepAlive));
	keepaliveval = keepAlive;

	connection_response = connect_async(sUrl.c_str(),port,keepAlive);

	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %s",mosquitto_strerror(connection_response)));
	MQTT_LOG_INFO_VAL((" TRYING TO CONNECT To  %s",sUrl.c_str()));
	if(connection_response != MOSQ_ERR_SUCCESS)
	{
		MQTT_LOG_INFO_VAL(("FAILED TO CONNECT :: %s",sUrl.c_str()));
		return connection_response;
	}

	MQTT_LOG_INFO_VAL((" STARTING THE NETWORK LOOP "));
	MQTT_LOG_INFO_VAL((" CURRENT LOOP STATUS ::  %d",NetworkLoopActive));

	if(false == NetworkLoopActive)
	{

		int response = loop_start();
		if(MOSQ_ERR_SUCCESS == response)
		{
			MQTT_LOG_INFO_VAL(("NETWORK LOOP STARTED"));
			NetworkLoopActive = true;
		}
		else
		{
			MQTT_LOG_INFO_VAL((" FAILED TO START ::  %d",response));
		}


	}
	else
	{
		MQTT_LOG_INFO_VAL((" NETWORK LOOP ALREADY STARTED"));
	}
	MQTT_LOG_INFO_VAL((" LOOP STATUS AFTER UPDATE ::  %d",NetworkLoopActive));

	oConACKTimer.start(*this,ConAckTimeOut*1000,Timer::Once);
	MQTT_LOG_INFO_VAL(("CONACK TIMER STARTED WITH :: %d Ms",ConAckTimeOut*1000));
	MQTT_LOG_INFO_VAL(("WAITING FOR CONACK "));


	return connection_response;
}

int mqtt_network_Manager::Publish(std::string& sTopic, std::string& sMsg,int& Mid, unsigned short Interval, unsigned short PubAckTimeout,unsigned short Occurence)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Publish() entered"));
	MQTT_LOG_INFO_VAL(("PUBLISH REQUEST RECIEVED"));
	MQTT_LOG_INFO_VAL(("TOPIC  :: %s",sTopic.c_str()));
	MQTT_LOG_INFO_VAL(("MSG :: %s",sMsg.c_str()));
	MQTT_LOG_INFO_VAL(("INTERVAL :: %d",Interval));
	MQTT_LOG_INFO_VAL(("OCCURRENCE :: %d",Occurence));
	MQTT_LOG_INFO_VAL(("TIMEOUT :: %d",PubAckTimeout));

	Publish_Interval = Interval;
	Publish_Occurence = Occurence;
	Publish_Timeout = PubAckTimeout;

	Remaining_Retry_Attempts = Publish_Occurence;

	I_mqtt_log_Manager*_poLogMgr = dynamic_cast<I_mqtt_log_Manager*>(_cpoMain->getHandler("I_mqtt_log_Manager"));
	MQTT_MANAGER_NULL_POINTER_CHECK_VAL(_poLogMgr);
	_poLogMgr->vWriteLogToFile(UPLOAD,"STAGE 1 : UPLOADING TO BROKER","UploadData.log",false);
	oPubACKTimer.start(*this,PubAckTimeout*1000,Timer::Once);
	MQTT_LOG_INFO_VAL(("PUBACK TIMER STARTED WITH :: %d Ms",PubAckTimeout*1000));
	message_retry_set(0);
	int response = publish(&Mid,sTopic.c_str(),sMsg.length(),sMsg.c_str(),PUBLISH_QOS,PUBLISH_RETAIN);
	
	MQTT_LOG_INFO_VAL(("WAITING FOR PUBACK WITH MID :: %d",Mid));
	MQTT_LOG_INFO_VAL((" PUBLISHING ...."));
	_poLogMgr->vWriteLogToFile(UPLOAD,"STAGE 2 : UPLOADED.WAITING FOR PUBACK","UploadData.log",false);

	MQTT_LOG_INFO_VAL(("PUBLIS RESPONSE :: %s",mosquitto_strerror(response)));

	if(MOSQ_ERR_SUCCESS != response)
	{
		MQTT_LOG_INFO_VAL(("FAILED TO PUBLISH !!!!!!!!!!!"));
		oPubACKTimer.stop();
		_poLogMgr->vWriteLogToFile(UPLOAD,"PUBLISH FAILED","UploadData.log",false);
		return response;
	}
	return response;

}

int mqtt_network_Manager::Subscribe(std::string& sTopic,unsigned short SubAckTimeout,int &Mid)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Publish() entered"));
	MQTT_LOG_INFO_VAL(("SUBSCRIBE REQUEST RECIEVED"));
	MQTT_LOG_INFO_VAL(("TOPIC  :: %s",sTopic.c_str()));

	int response = subscribe(&Mid,sTopic.c_str(),SUBSCRIBE_QOS);

	MQTT_LOG_INFO_VAL(("SUBSCRIBE RESPONSE :: %s",mosquitto_strerror(response)));

	if(MOSQ_ERR_SUCCESS != response)
	{
		MQTT_LOG_INFO_VAL(("FAILED TO SUBSCRIBE !!!!!!!!!!!"));
		return response;
	}

	MQTT_LOG_INFO_VAL((" SUBSCRIBING ...."));
	oSubAckTimer.start(*this,SubAckTimeout*1000,Timer::Once);
	MQTT_LOG_INFO_VAL(("SUBACK TIMER STARTED WITH :: %d Ms",SubAckTimeout*1000));
	MQTT_LOG_INFO_VAL(("WAITING FOR SUBACK WITH MID :: %d",Mid));
	return response;
}

void mqtt_network_Manager::ActionOnPUBACKTimerExpired()
{
	MQTT_LOG_INFO_VAL(("PUBLISH FAILED !!!!!!"));
	MQTT_LOG_INFO_VAL(("REAMINING PUBLISH ATTEMPTS :: %d",Remaining_Retry_Attempts));
	if(Remaining_Retry_Attempts > 0 )
	{
		if(Publish_Interval!= 0)
		{
			oPublishIntervalTimer.start(*this,Publish_Interval*1000,Timer::Once);
			MQTT_LOG_INFO_VAL(("PUBLISH RETRY INTERVAL TIMER STARTED WITH %d Ms",Publish_Interval*1000));
		}
		else
		{
			ActionOnPUBINTimerExpired();
		}
	}
	else
	{
		MQTT_LOG_INFO_VAL(("MAX RETTY ATTEMPT REACHED..DISCONNECTING"));
		MQTT_LOG_INFO_VAL(("DISCONNECTING DUE TO PUBLISH FAIL!"));
		message_retry_set(UINT16_MAX);
		DisconnectionReason = PUBLISH_FAIL;
		InternalDisconnect();
	}

	bReconnectedDueToPublishFail = true;
}

void mqtt_network_Manager::ActionOnSUBACKTimerExpired()
{
	MQTT_LOG_INFO_VAL(("SUBSCRIBE FAILED !!!!!!"));
	MQTT_LOG_INFO_VAL(("DISCONNECTING DUE TO SUBSCRIBE FAIL!"));
	DisconnectionReason = SUBSCRIBE_FAIL;
	InternalDisconnect();

}

bool mqtt_network_Manager::bIsTLSFatalError()
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::bIsTLSFatalError() entered"));

	std::string sErrorMessage;
	stunnel_log_parser oLogParser;
	oLogParser.Dump_Stunnel_Log_To_DLT();
	stunnel_Handler oStullenHandler;

	bool bIsFatalError = oLogParser.bIsFatalError("mqtt_stunnel.log",oTLS_Error_List,sErrorMessage);

	if(true == bIsFatalError)
	{
		MQTT_LOG_INFO_VAL((" SSL FATAL ERROR"));
		//MQTT_LOG_INFO_VAL((" SSL ERROR :: %s",sErrorMessage.c_str()));
		MQTT_LOG_INFO_VAL((" DISCONNECTED DUE TO SSL FATAL ERROR. NO MORE CONNECTION REQUEST POSSIBLE IN THIS POWER CYCLE"));
		stunnel_Handler oHandler;
		oHandler.reopenLogFile();
		std::stringstream ss;
		ss << "TLS FATAL ISSUE OCCURED - NO MORE CONNECTION ATTEMPTS";
		oHandler.addHeaderBeforeReopen(ss.str());
	}

	return bIsFatalError;
}

int mqtt_network_Manager::Reinitialize(bool Cleansession)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Reinitialize() entered"));
	int response = reinitialise(sNaviUnitId.c_str(),Cleansession);

	if(response != MOSQ_ERR_SUCCESS)
	{
		MQTT_LOG_INFO_VAL((" FAILED TO RE-INITIALISE"));
		return response;
	}
	int mqtt_ver = MQTT_PROTOCOL_V311;
	void* ver = &mqtt_ver;
	opts_set(MOSQ_OPT_PROTOCOL_VERSION,ver);
	MQTT_LOG_INFO_VAL(("SETTING PROTOCOL VERSION TO MQTT_PROTOCOL_V311"));


	MQTT_LOG_INFO_VAL(("SETTING MAX RECONNECT DELAY "));
	reconnect_delay_set(UINT32_MAX,UINT32_MAX,false);

	MQTT_LOG_INFO_VAL(("SETTING MAX MESSAGE RETRY DELAY "));
	message_retry_set(0);

	MQTT_LOG_INFO_VAL((" RE-INITIALISED"));
	return response;
}


void mqtt_network_Manager::SendResponse(MqttNetworkResponse *Response)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::SendResponse(MqttNetworkResponse Response) entered"));
	I_mqtt_Manager*_poMqttMgr = dynamic_cast<I_mqtt_Manager*>(_cpoMain->getHandler("I_mqtt_Manager"));
	MQTT_MANAGER_NULL_POINTER_CHECK(_poMqttMgr);
	_poMqttMgr->bPostMessage(Response);

}

int mqtt_network_Manager::Disconnect()
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Disconnect() entered"));
	MQTT_LOG_INFO_VAL(("DISCONNECT REQUEST RECIEVED"));

	int response = MOSQ_ERR_UNKNOWN;

	if(true == Connected)
	{
		DisconnectionReason = EXTRANAL;
		response = disconnect();
	}
	else
	{
		MQTT_LOG_INFO_VAL(("ALREADY DISCONNECTED"));
		response = MOSQ_ERR_SUCCESS;
	}

	return response;
}

void mqtt_network_Manager::LoadTLSErrorList()
{
	MQTT_LOG_INFO_VAL(("LOADING TLS ERROS STRING"));
	I_mqtt_JSONParser *poJsonInst = dynamic_cast<I_mqtt_JSONParser*>(_cpoMain->getHandler("I_mqtt_JSONParser"));
	MQTT_MANAGER_NULL_POINTER_CHECK(poJsonInst);
	poJsonInst->oGetErrorStringList(oTLS_Error_List);
	MQTT_LOG_INFO_VAL(("TLS ERROS STRINGS LOADED"));
	MQTT_LOG_INFO_VAL((" =======ERROR STRINGS======== "));
	for(auto unsigned list_index = 0 ; list_index < oTLS_Error_List.size();list_index++ )
		MQTT_LOG_INFO_VAL((" %s",oTLS_Error_List.at(list_index).sError_String.c_str()));
	MQTT_LOG_INFO_VAL((" =============================== "));
}


bool mqtt_network_Manager::ConnectionPreCheck(int &connection_response)
{
	MQTT_LOG_INFO_VAL(("CONNECTION PRE-CHECK "));

	if(Connected == true)
	{
		MQTT_LOG_INFO_VAL(("ALREADY CONNECTED "));
		connection_response = MOSQ_ERR_CONN_PENDING;
		return false;
	}

	if(bIpServiceAvaibility == false)
	{
		MQTT_LOG_INFO_VAL(("NO_INTERNET"));
		connection_response = MOSQ_ERR_EAI;
		return false;
	}

	return true;
}

void mqtt_network_Manager::stop_loop(bool force)
{
	MQTT_LOG_INFO_VAL((" STOPING THE NETWORK LOOP "));
	MQTT_LOG_INFO_VAL((" FORCE ? :: %d",force));
	MQTT_LOG_INFO_VAL((" CURRENT LOOP STATUS ::  %d",NetworkLoopActive));

	if(true == NetworkLoopActive)
	{
		NetworkLoopActive = false;
		loop_stop(force);
	}
	else
	{
		MQTT_LOG_INFO_VAL(("NETWORK LOOP IS NOT ACTIVE"));
	}

	MQTT_LOG_INFO_VAL((" LOOP STATUS AFTER UPDATE ::  %d",NetworkLoopActive));
}

int mqtt_network_Manager::RetryPublish()
{
	MQTT_LOG_INFO_VAL((" PUBLISH RETRY  "));
	MQTT_LOG_INFO_VAL((" REAMINING PUBLISH ATTEMPTS :: %d",Remaining_Retry_Attempts));
	MQTT_LOG_INFO_VAL((" PUBLISH ATTEMPT :: %d",(Publish_Occurence - Remaining_Retry_Attempts) + 1));
	Remaining_Retry_Attempts--;
	loop(0,1);
	MQTT_LOG_INFO_VAL(("REAMINING PUBLISH ATTEMPTS NOW :: %d",Remaining_Retry_Attempts));
	oPubACKTimer.start(*this,(Publish_Timeout)*1000,Timer::Once);
	MQTT_LOG_INFO_VAL(("PUBACK TIMER STARTED WITH :: %d Ms",Publish_Timeout*1000));
	MqttPublishResonse *resonse =  new MqttPublishResonse(PUBLISH);
	resonse->setResponseCode(PUBLISH_RETRY_ATTEMPTED);
	resonse->setMid(0);
	SendResponse(resonse);
	return MOSQ_ERR_SUCCESS;
}

void mqtt_network_Manager::ActionOnPUBINTimerExpired()
{
	MQTT_LOG_INFO_VAL(("PUBINT TIMER EXPIRED"));
	RetryPublish();
}

void mqtt_network_Manager::InternalDisconnect()
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::InternalDisconnect() entered"));
	MQTT_LOG_INFO_VAL(("INTERNAL DISCONNECT REQUEST RECIEVED"));

	int response = MOSQ_ERR_UNKNOWN;

	if(true == Connected)
	{
		response = disconnect();
	}
	else
	{
		MQTT_LOG_INFO_VAL(("ALREADY DISCONNECTED"));
		response = MOSQ_ERR_SUCCESS;
	}
}

void mqtt_network_Manager::SetIpServiceAvailability(bool bIpServiceStatus)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::SetIpServiceAvailability() entered"));
	MQTT_LOG_INFO_VAL(("IP CONNECTIVITY SERVICE CHANGED TO :: %d",bIpServiceStatus));
	bIpServiceAvaibility = bIpServiceStatus;
	if(false == bIpServiceAvaibility)
	{
		MQTT_LOG_INFO_VAL(("CONNECTION STATUS :: %d",Connected));
		if(true == Connected)
		{
			MqttConnectionResponse *response = new MqttConnectionResponse(CONNECTION);
			response->setConnectionResponseCode(CONNECTION_DISCONNECTED_DUE_TOCONNECTIVITY_FAILED);
			SendResponse(response);
			Connected = false;
			message_retry_set(UINT16_MAX);
			stop_loop(true);
			MQTT_LOG_INFO_VAL((" REMAINING ATTEMPTS WHEN CONNECTION WAS LOST :: %d ",Remaining_Retry_Attempts));
			if (Remaining_Retry_Attempts > 0)
			{
				MQTT_LOG_INFO_VAL((" CONNECTIVITY LOST WHILE PUBLISH IN PROGRESS"));
				bReconnectedDueToPublishFail = true;
			}
			Remaining_Retry_Attempts = 0;
			oPubACKTimer.stop();
			oPublishIntervalTimer.stop();
			
		}
	}
}

int mqtt_network_Manager::ReConnect(std::string &sUrl, unsigned port, unsigned short keepAlive, unsigned short ConAckTimeOut)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::SetIpServiceAvailability() entered"));
	MQTT_LOG_DEBUG(("mqtt_network_Manager::Connect() entered"));
	MQTT_LOG_INFO_VAL(("CONNECT REQUEST RECIEVED"));
	int connection_response = MOSQ_ERR_EAI;

	if(false == ConnectionPreCheck(connection_response))
	{
		MQTT_LOG_INFO_VAL(("CONNECT REQUEST REFUSED"));
		return connection_response;
	}

	MQTT_LOG_INFO_VAL(("URL  :: %s",sUrl.c_str()));
	MQTT_LOG_INFO_VAL(("PORT :: %d",port));
	MQTT_LOG_INFO_VAL(("KEEPALIVE :: %d",keepAlive));
	keepaliveval = keepAlive;

	connection_response = reconnect_async();

	MQTT_LOG_INFO_VAL(("CONNECTION RESPONSE :: %s",mosquitto_strerror(connection_response)));
	MQTT_LOG_INFO_VAL((" TRYING TO CONNECT To  %s",sUrl.c_str()));
	if(connection_response != MOSQ_ERR_SUCCESS)
	{
		MQTT_LOG_INFO_VAL(("FAILED TO CONNECT :: %s",sUrl.c_str()));
		return connection_response;
	}

	MQTT_LOG_INFO_VAL((" STARTING THE NETWORK LOOP "));
	MQTT_LOG_INFO_VAL((" CURRENT LOOP STATUS ::  %d",NetworkLoopActive));

	if(false == NetworkLoopActive)
	{

		int response = loop_start();
		if(MOSQ_ERR_SUCCESS == response)
		{
			MQTT_LOG_INFO_VAL(("NETWORK LOOP STARTED"));
			NetworkLoopActive = true;
		}
		else
		{
			MQTT_LOG_INFO_VAL((" FAILED TO START ::  %d",response));
		}


	}
	else
	{
		MQTT_LOG_INFO_VAL((" NETWORK LOOP ALREADY STARTED"));
	}
	MQTT_LOG_INFO_VAL((" LOOP STATUS AFTER UPDATE ::  %d",NetworkLoopActive));

	oConACKTimer.start(*this,ConAckTimeOut*1000,Timer::Once);
	MQTT_LOG_INFO_VAL(("CONACK TIMER STARTED WITH :: %d Ms",ConAckTimeOut*1000));
	MQTT_LOG_INFO_VAL(("WAITING FOR CONACK "));


	return connection_response;
}

void mqtt_network_Manager::UpdateDisconnectionReasonToMqttLogFile(	std::string& sLogMsg, int serverresonsecode)
{
	I_mqtt_log_Manager*_poLogMgr = dynamic_cast<I_mqtt_log_Manager*>(_cpoMain->getHandler("I_mqtt_log_Manager"));
	MQTT_MANAGER_NULL_POINTER_CHECK(_poLogMgr);
	std::stringstream ss;
	ss<<sLogMsg;
	ss<<"("<<serverresonsecode<<")";
	_poLogMgr->vWriteLogToFile(MQTT_APPLICATION,ss.str().c_str(),"mqtt_network_log.txt",false);
}

void mqtt_network_Manager::AdditionalStepsOnDisconnection(unsigned ConnectionResponseCode)
{
	MQTT_LOG_DEBUG(("mqtt_network_Manager::AdditionalStepsOnDisconnection() entered"));
	switch(ConnectionResponseCode)
	{
		case CONNECTION_FAILED_BROKER_DISCONNECTED:
		{
			//Stop All the Timers
			if(TRUE == oSubAckTimer.isActive())
			{
				MQTT_LOG_INFO_VAL((" STOPPING SUB-ACK TIMER"));
				oSubAckTimer.stop();
			}

			if(TRUE == oPubACKTimer.isActive())
			{
				MQTT_LOG_INFO_VAL((" STOPPING PUB-ACK TIMER"));
				MQTT_LOG_INFO_VAL((" CONNECTION DISCONNECTED WHILE WAITING FOR PUBACK. PUBLISH SHALL BE RETRIED NEXT CONNECTION"));
				oPubACKTimer.stop();
				bReconnectedDueToPublishFail = true;
			}
		}
		break;
	}
}

void mqtt_network_Manager::PublishRetryOnConnection()
{
	MQTT_LOG_INFO_VAL(("ONE PUBLISH FAILED ITEM IS PENDING"));
	message_retry_set(0);
	bReconnectedDueToPublishFail = false;
	Remaining_Retry_Attempts = Publish_Occurence;
	MQTT_LOG_INFO_VAL(("REAMINING PUBLISH ATTEMPTS CONFIGURED :: %d",Remaining_Retry_Attempts));	
	oPendingItemUploadTimer.start(*this,PENDING_PUBLISH_ITEM_POST_DELAY,Timer::Once);
	MQTT_LOG_INFO_VAL(("FAILED PUBLISH ITEM WILL BE POSTED AFTER :: %d",PENDING_PUBLISH_ITEM_POST_DELAY));
}
